package org.briarproject.mailbox.core.setup

import com.fasterxml.jackson.module.kotlin.jacksonObjectMapper
import com.fasterxml.jackson.module.kotlin.readValue
import io.ktor.client.call.body
import io.ktor.client.request.put
import io.ktor.client.statement.HttpResponse
import io.ktor.client.statement.bodyAsText
import io.ktor.http.HttpStatusCode
import kotlinx.coroutines.runBlocking
import org.briarproject.mailbox.core.TestUtils.assertJson
import org.briarproject.mailbox.core.TestUtils.assertTimestampRecent
import org.briarproject.mailbox.core.server.IntegrationTest
import org.briarproject.mailbox.core.settings.MailboxVersion
import org.junit.jupiter.api.Test
import kotlin.test.assertEquals
import kotlin.test.assertNull
import kotlin.test.assertTrue

class SetupManagerTest : IntegrationTest() {

    @Test
    fun `setup request gets rejected when using non-setup token`() = runBlocking {
        // initially, there's no setup and no owner token
        db.read { txn ->
            assertNull(setupManager.getSetupToken(txn))
            assertNull(setupManager.getOwnerToken(txn))
        }
        // owner token gets rejected
        assertEquals(
            HttpStatusCode.Unauthorized,
            httpClient.put("$baseUrl/setup") {
                authenticateWithToken(ownerToken)
            }.status
        )

        // now we set the owner token which still gets rejected
        setupManager.setToken(null, ownerToken)
        assertEquals(
            HttpStatusCode.Unauthorized,
            httpClient.put("$baseUrl/setup") {
                authenticateWithToken(ownerToken)
            }.status
        )

        // now we set the setup token, but use a different one for the request, so it gets rejected
        setupManager.setToken(token, null)
        assertEquals(
            HttpStatusCode.Unauthorized,
            httpClient.put("$baseUrl/setup") {
                authenticateWithToken(ownerToken)
            }.status
        )
    }

    @Test
    fun `setup request clears setup token and sets new owner token`() = runBlocking {
        // set a setup-token
        setupManager.setToken(token, null)

        // we are not yet set up
        assertNull(db.read { txn -> setupManager.getOwnerToken(txn) })

        // use it for setup PUT request
        val response: SetupResponse = httpClient.put("$baseUrl/setup") {
            authenticateWithToken(token)
        }.body()
        // setup token got wiped and new owner token from response got stored
        db.read { txn ->
            assertNull(setupManager.getSetupToken(txn))
            assertEquals(setupManager.getOwnerToken(txn), response.token)
        }
        // owner connection got registered
        assertTimestampRecent(metadataManager.ownerConnectionTime.value)
        // setup token can no longer be used
        assertEquals(
            HttpStatusCode.Unauthorized,
            httpClient.put("$baseUrl/setup") {
                authenticateWithToken(token)
            }.status
        )
    }

    @Test
    fun `authentication doesn't work with empty string`() = runBlocking {
        // use it for setup PUT request
        val response: HttpResponse = httpClient.put("$baseUrl/setup") {
            authenticateWithToken("")
        }
        assertEquals(HttpStatusCode.Unauthorized, response.status)
    }

    @Test
    fun `setup request returns supported versions`() = runBlocking {
        // set a setup-token
        setupManager.setToken(token, null)

        // use it for setup PUT request
        val response: SetupResponse = httpClient.put("$baseUrl/setup") {
            authenticateWithToken(token)
        }.body()

        assertTrue(response.serverSupports.contains(MailboxVersion(1, 0)))
    }

    @Test
    fun `setup request returns supported versions in correct format`() = runBlocking {
        // set a setup-token
        setupManager.setToken(token, null)

        // use it for setup PUT request
        val rawResponse: HttpResponse = httpClient.put("$baseUrl/setup") {
            authenticateWithToken(token)
        }

        // read object to retrieve owner token generated by SetupManager
        val response: SetupResponse = jacksonObjectMapper().readValue(rawResponse.bodyAsText())

        // make sure the response is in the expected format, allowing whitespace variations
        assertJson(
            HttpStatusCode.Created,
            """{ "token": "${response.token}", "serverSupports": [ {"major": 1, "minor": 0} ] }""",
            rawResponse
        )
    }

}
